------------------------------------------------------
-- SKL_Exporter.ms
--
-- Elite Force 2 SKL Exporter for 3DS Max 5 Version 1.1
-- Copyright (C) Neil Davis, 2003
-- mailto:senkusha@yahoo.com
-- 
-- Many thanks to Chris Cookson for writing so many script from which to learn maxscript from.
-- And many thanks to Ritual for putting together one hell of a game.
-- 
-- License:
-- --------
-- 
-- Copyright (c) 2003 Neil Davis
-- 
-- Permission is hereby granted, free of charge, to any
-- person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software
-- without restriction, including without limitation the rights
-- to use, copy, modify, merge, publish, distribute, sublicense,
-- and/or sell copies of the Software, and to permit persons to
-- whom the Software is furnished to do so, subject to the following
-- conditions:
-- 
-- The above copyright notice and this permission notice shall be
-- included in all copies or substantial portions of the Software.
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
-- EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
-- OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
-- IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
-- ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
-- CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
-- WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
-- 
-- 
-- History:
-- --------
-- 
-- Version	1.0: First public release.
-- Version	1.1: Fixed animations not being exported correctly
--				 Fixed Remove* buttons of user interface. 
--				 Added tentative ProgressBar
--				 Fixed Animation spinners so that the most frames of animation exported was 100
--				 Note: The maximum number of frames allowed to be exported by sklout.dle is 2048
--					   To even get near this number, the script must increase the heapsize used by maxscript to nearly 30 megs!
--			1.2: Added some (i.e. only bones and frames at the moment) limit checking
--				 Added check to ensure that the morpher.dlm that shipped with the ubertools can not be used		
--				 Maxscript heapsize no longer needs to be increased to export large amounts of frames
--					
--
--
-- ToDo:
-- -----
-- 
-- 1) DONE: Add tag names to material export 
-- 2) DONE: Add LEGBONE support
-- 3) DONE: Add morph export support
-- 4) DONE: add a progress bar
------------------------------------------------------

global VERSION = 1

utility SKL_Export "SKL Exporter"
(
	local objSelectionList = #()
	local legSelectionList = #()
	local fp
		
	local objList = #()
	local boneList = #()
	local legList = #()
	local parentList = #()
	
	local model
	local numframes
	
	local morphSkinObj
	local morphModObj
	local morphBaseObj
	
	local morphTargets = #()
	local numMorphTargets
	
	group "About:"
	(
		label titleLabel "Elite Force 2 SKL Exporter v1.1"
		HyperLink addy "by Neil Davis" align:#center address:"mailto:senkusha" color:(color 0 0 170) hoverColor:(color 170 0 0)
	)
	
	group "Objects:"
	(
		multiListBox objListBox height:10
		button addObj "Add Selected" 
		button removeObj "Remove Selected" 
	)
	
	on addObj pressed do
	(
		local names = for obj in selection collect obj.name
		objSelectionList = objListBox.items
		for i = 1 to names.count do
			append objSelectionList names[ i ]
		objListBox.items = objSelectionList
	)

	on removeObj pressed do
	(
		local mlbSelection = objListBox.selection as array
		if mlbSelection.count > 0 then
		(
			objSelectionList = objListBox.items

			i = mlbSelection.count
			while i != 0 do
			(
				deleteItem objSelectionList mlbSelection[ i ]
				i -= 1
			)
			
			objListBox.items = objSelectionList
		)
	)
	
	group "Leg Bones:"
	(
		checkbox useLegBones "Use Leg Bones..."
		multiListBox legListBox height:5 enabled:false
		button addLeg "Add Selected" enabled:false
		button removeLeg "Remove Selected" enabled:false
	)
	
	on addLeg pressed do
	(
		local names = for obj in selection collect obj.name
		legSelectionList = legListBox.items
		for i = 1 to names.count do
			append legSelectionList names[ i ]
		legListBox.items = legSelectionList
	)
	
	on removeLeg pressed do
	(
		local mlbSelection = legListBox.selection as array
		if mlbSelection.count > 0 then
		(
			legSelectionList = legListBox.items
			
			i = mlbSelection.count
			while i != 0 do
			(
				deleteItem legSelectionList mlbSelection[ i ]
				i -= 1
			)
					
			legListBox.items = legSelectionList
		)
	)
		
	on useLegBones changed state do
	(
		if useLegBones.checked == true then
		(
			legListBox.enabled = true
			addLeg.enabled = true
			removeLeg.enabled = true
		)
		else if useLegBones.checked == false then
		(
			legListBox.enabled = false
			addLeg.enabled = false
			removeLeg.enabled = false
		)
	)

	
	
	group "Morph Targets:"
	(
		checkbox exportMorphTargets "Export Morph Targets..."
		pickButton pickMorphObj "Select Morph Object..." enabled:false
	)
	
	on exportMorphTargets changed state do
	(
		if exportMorphTargets.checked == true then
		(
			pickMorphObj.enabled = true
		)
		else if exportMorphTargets.checked == false then
		(
			pickMorphObj.enabled = false
		)
	)
	
	on pickMorphObj picked obj do
	(
		-- make sure the object has a skin and morpher mod applied to it
		morphSkinObj = obj.modifiers[ #Skin ]
		if morphSkinObj == undefined then
		(
			messagebox "No valid Skin modifier on this object" caption:"ERROR" beep:true
			return false	
		)
		morphModObj = obj.modifiers[ #morpher ]
		if morphModObj == undefined then
		(
			messagebox "No valid Morpher modifier on this object.\nAre you sure your not using the Morpher Modifier that came with the UberTools." caption:"ERROR" beep:true
			return false
		)
		if (IsValidMorpherMod morphModObj) == false then
		(
			messagebox "No valid Morpher modifier on this object.\nAre you sure your not using the Morpher Modifier that came with the UberTools." caption:"ERROR" beep:true
			return false
		)
		pickMorphObj.text = obj.name
		morphBaseObj = obj
	)
	
	local timeInterval = animationRange
	local startTime = (timeInterval.start as integer) / ticksPerFrame
	local endTime = (timeInterval.end as integer) / ticksPerFrame
	
	group "Export:"
	(
		radioButtons exportRadio labels:#("Reference Pose", "Animation Sequence") default:1
		spinner firstFrameSpinner "First Frame:" type:#integer 	enabled:false range:[startTime,endTime,0]
		spinner lastFrameSpinner  "Last Frame:"  type:#integer  enabled:false range:[startTime,endTime,0]
		button exportButton "Export SKL..."	tooltip:"Export SKL model"
	)
	
	on exportRadio changed state do
	(
		if state == 2 then
		(
			firstFrameSpinner.enabled = true
			lastFrameSpinner.enabled = true
		)
		else
		(
			firstFrameSpinner.enabled = false
			lastFrameSpinner.enabled = false
		)
	)

	fn WriteFloat6 x fstream=
	(
		if false then
		(
			format "% " x to:fstream
		)
		else
		(
			local ipart = x as integer
			local fpart = abs (x - ipart)
			ipart = abs ipart
			if x < 0.0 do format "-" to:fstream
			
			local scaledStr = ((1000000.0 * fpart) as integer) as string
			local startPos = amax 1 (7 - scaledStr.count)
			local len = amax 0 (7 - startPos)
			local fracStr = replace "000000" startPos len scaledStr
			format "%.% " ipart fracStr to:fstream
		)
	)
	
	include "skl_data.ms"
	
	fn InitStructures = 
	(
		model = loadsingle()
		frames = loadframe()

		for i = 1 to model.usedbone.count do
			model.usedbone[ i ] = false;
	
		ok
	)
	
	struct tagInfo
	(
		ID,
		name
	)
	
	fn GetMaterials =
	(		
		local a
		local b
					
		for i = 1 to model.numfaces do
		(
			a = i
			for j = 1 to model.nummaterials do
			(
				b = j
				if model.materials[ j ] == model.faces[ i ].id then
				(
					exit	
				)
			)
			if b == model.nummaterials then
			(
				append model.materials model.faces[ i ].id
				model.nummaterials += 1
			)	
		)
		
		model.nummaterials -= 1
		for m = 1 to model.nummaterials do
		(
			model.materials[ m ] -= 1
		)
				
		-- setup material names for export
		tagObjs = for obj in objects where (((findstring obj.name "tag") == 1 )or ((findstring obj.name "origin") == 1)) collect obj
			
		local tags = #()
		
		for t = 1 to tagObjs.count do
		(
			local tag = tagInfo()
			
			tag.id = getFaceMatID tagObjs[ t ] 1
			tag.name = tagObjs[ t ].name
			
			append tags tag
		)
						
		for m = 1 to model.nummaterials do
		(
			local isTagMat = false
			-- see if the current ID belongs to a tag
			for t = 1 to tags.count do
			(
				if model.materials[ m ] == tags[ t ].id then
				(
					isTagMat = true
					append model.materialnames (copy tags[ t ].name)
				)
			)
			
			if isTagMat == false then
			(
				local name = "material" + (model.materials[ m ] as string)
				append model.materialnames (copy name)
			)
		)
	)
	
	fn GetSkinVerts skinObj mesh = 
	(
		
		for v = 1 to mesh.numverts do
		(
			local vert 			= loadvertex()
			local numBones 		= skinOps.GetVertexWeightCount skinObj v
			local vert_pos 		= getVert mesh v
			
			if numBones > MAX_BLENDBONES then
			(
				numBones = MAX_BLENDBONES
			)
			vert.numbones = numBones
			for b = 1 to numBones do
			(
				local curBlend	= blendvert()
				curBlend.bone 	= skinOps.GetVertexWeightBoneID skinObj v b
				local boneName 	= skinOps.GetBoneName skinObj curBlend.bone b
				local bone
				for i = 1 to boneList.count do
				(
					if boneList[ i ].name == boneName then
					(
						curBlend.bone = (i - 1)
						bone = boneList[ i ]
					)
				)
				local boneTM = bone.transform as matrix3
							
				if curBlend.bone == -1 then
				(
					 errorMsg = "Found vertex attached to unused bone"
					 throw errorMsg
				)
				if model.usedbone[ curBlend.bone ] == false then
				(
					model.usedbone[ curBlend.bone ] = true
					model.numusedbone += 1
				)
				
				local x = length boneTM.row1
				local y = length boneTM.row2
				local z = length boneTM.row3
				local s = [x, y, z] as point3
							
				local invBoneTM = (inverse boneTM)
				local p = vert_pos * invBoneTM
			
				x = p.x * s.x
				y = p.y * s.y
				z = p.z * s.z
				curBlend.offset = [x, y, z]
				curBlend.weight = skinOps.GetVertexWeight skinObj v b
						
				append vert.blend curBlend
			)
			append model.verts vert
		)
	)
	
	fn GetNonSkinnedVerts mesh name=
	(
		for v = 1 to mesh.numverts do
		(
			progressUpdate (100.0 * v / (mesh.numverts + mesh.numfaces))
			
			local vert 		= loadvertex()
			local curBlend 	= blendvert()
			local vert_pos 	= getVert mesh v
			local bone
			vert.numbones = 1
			
			for i = 1 to boneList.count do
			(
				if boneList[ i ].name == name then
				(
					curBlend.bone = (i - 1)
					bone = boneList[ i ]
				)
			)
			
			local boneTM = bone.transform as matrix3
			
			if curBlend.bone == -1 then
			(
				errorMsg = "Found vertex attached to unused bone"
				throw errorMsg
			)
			if model.usedbone[ curBlend.bone ] == false then
			(
				model.usedbone[ curBlend.bone ] = true
				model.usedbone += 1 
			)
			
			local x = length boneTM.row1
			local y = length boneTM.row2
			local z = length boneTM.row3
			local s = [x, y, z] as point3
			
			local invBoneTM = (inverse boneTM)
			local p = vert_pos * invBoneTM
			
			x = p.x * s.x
			y = p.y * s.y
			z = p.z * s.z
			curBlend.offset = [x, y, z]
			curBlend.weight = 1.0
			
			append vert.blend curBlend
			
			append model.verts vert
		)
	)
	
	fn AddObject obj =
	(
		local myMesh = snapShotAsMesh obj
		local myTriMesh = obj.mesh
		local curObject = loadobject()
				
		curObject.startvert = model.numverts;
		curObject.startface = model.numfaces;
		
		curObject.name = obj.name
		curObject.numverts = myTriMesh.numverts
		curObject.numfaces = myTriMesh.numfaces

		model.hasmapping = (myTriMesh.numtverts > 0)
		
		-- find skin modifier
		local skinObj = obj.modifiers[ 1 ]
		if classOf skinObj != Skin then
		(
			GetNonSkinnedVerts myMesh obj.name
		)
		else
		(
			max modify mode
			modPanel.SetCurrentObject skinObj	
			skinObj.rigid_vertices = true
			GetSkinVerts skinObj myMesh
		)
		
		-- grab faces
		for f = 1 to myTriMesh.numfaces do
		(
			local smoothingGroup 	= getFaceSmoothGroup myTriMesh f
			local normal 			= getFaceNormal myTriMesh f
			local index 			= getFace myTriMesh f
			local curTriangle 		= loadtriangle()
			
			for i = 1 to 3 do
			(
				local curLoadVert =  model.verts[ curObject.startvert + index[ i ] ]
				
								
				if smoothingGroup != 0 then
				(
					curLoadVert.normal = getNormal myTriMesh index[ i ]
				)
				else
				(
					local x = normal.x
					local y = normal.y
					local z = normal.z
					curLoadVert.normal = [x, y, z]
				)
				
				local curFaceVertex = loadfacevertex()
				curFaceVertex.vertindex = index[ 4 - i ] + curObject.startvert
				if model.hasmapping then
				(
					--loadfacevert->s = mesh->tVerts[ mesh->tvFace[ f ].t[ 2 - j ] ].x;
					--loadfacevert->t = 1.0f - mesh->tVerts[ mesh->tvFace[ i ].t[ 2 - j ] ].y;
					
					local texIndex = getTVFace myTriMesh f
					local texCoord = getTVert myTriMesh texIndex[ 4 - i ]
					
					curFaceVertex.s = texCoord.x
					curFaceVertex.t = (1.0 - texCoord.y)
				)
				else
				(
					curFaceVertex.s = 0.0
					curFaceVertex.t = 0.0
				)
				
				append curTriangle.verts curFaceVertex
			)
			--uvout adds 1 so add 1 here
			curTriangle.id = (getFaceMatID myTriMesh f) + 1
						
			append model.faces curTriangle
		)
	
		append model.objects curObject
		model.numverts += curObject.numverts
		model.numfaces += curObject.numfaces
		model.numobjects += 1
	)
	
	fn CollectObjects = 
	(
		boneList = for obj in objects where (obj.isHidden == false) collect obj
		
		if boneList.count > 256 then
		(
			errorMsg = "Exceded max number of bones: 256"
			throw errorMsg
		)
		
		for i = 1 to boneList.count do
		(
			local curname = boneList[ i ].name
			if model.usedbone[ i ] == false and ((findstring curname "tag") == 1 ) then 
			(
				model.usedbone[ i ] = true
				model.numusedbone += 1
			)
		)
		
		-- collect objects
		for i = 1 to objSelectionList.count do
		(
			for j = 1 to boneList.count do
			(
				if boneList[ j ].name == objSelectionList[ i ] then
				(
					local progStr = "Exporting " + boneList[ j ].name + "..."
					progressStart progStr
					AddObject boneList[ j ]
					progressEnd()
				)		
			)
		)
	)
	
	fn AddBoneFrames fp = 
	(
		local firstFrame = 0
		local lastFrame = 0
		local curFrame = 0
							
		-- baseframe export
		if exportRadio.state == 2 then
		(
			firstFrame = firstFrameSpinner.value
			lastFrame = lastFrameSpinner.value
		)
		-- ensure that the first frame is smaller than the last frame
		if firstFrame > lastFrame then
		(
			messagebox "First Frame is after Last Frame.  Exporting baseframe instead at frame 0." caption:"Warning" beep:true
			firstFrame = 0
			lastFrame = 0
		)
		
		numFrames = (lastFrame+1) - firstFrame
		if numFrames > 8192 then
		(
			errorMsg = "Exceded max number of frames: 8192"
			throw errorMsg
		)
		
		format "NUMFRAMES %\n\n" numFrames to:fp
		for f = firstFrame to (lastFrame) do
		(
			local progStr = "Converting frame " + (f as string) + "..."
					
			progressStart progStr
			progressUpdate (100.0 * f / numFrames )
			
			format "FRAME %\n" curFrame to:fp
			at time f
			(
				for b = 1 to boneList.count do
				(
					local pos
					local rot
					
					local curBoneFrame = loadbone()
					curBoneFrame.offset = boneList[ b ].transform.translationPart
					local row1 = boneList[ b ].transform.row1
					local row2 = boneList[ b ].transform.row2
					local row3 = boneList[ b ].transform.row3
					curBoneFrame.matrix = matrix3 row1 row2 row3 [0,0,0]
					
					format "BONE %\n" (b-1) to:fp
					format "OFFSET " to:fp
						(WriteFloat6 curBoneFrame.offset.x fp)
						(WriteFloat6 curBoneFrame.offset.y fp)
						(WriteFloat6 curBoneFrame.offset.z fp)
					format "\n" to:fp
					format "X " to:fp
						(WriteFloat6 curBoneFrame.matrix.row1.x fp)
						(WriteFloat6 curBoneFrame.matrix.row1.y fp)
						(WriteFloat6 curBoneFrame.matrix.row1.z fp)
					format "\n" to:fp
					format "Y " to:fp
						(WriteFloat6 curBoneFrame.matrix.row2.x fp)
						(WriteFloat6 curBoneFrame.matrix.row2.y fp)
						(WriteFloat6 curBoneFrame.matrix.row2.z fp)
					format "\n" to:fp
					format "Z " to:fp
						(WriteFloat6 curBoneFrame.matrix.row3.x fp)
						(WriteFloat6 curBoneFrame.matrix.row3.y fp)
						(WriteFloat6 curBoneFrame.matrix.row3.z fp)
					format "\n" to:fp
					
					format "\n" to:fp
				)
			)
			curFrame += 1
		)
		progressEnd()
	)
	
	fn GetBoneParents =
	(
		parentList.count = boneList.count
	
		for b = 1 to boneList.count do
		(
			boneParent = boneList[ b ].parent
			if boneParent == undefined then
			(
				parentList[ b ] = -1
			)
			else
			(
				local boneParentName = boneParent.name
				for i = 1 to boneList.count do
				(
					if boneParentName == boneList[ i ].name then
					(
						parentList[ b ] = (i - 1)
						exit
					)
				)
			)
		)
	)
	
	fn GetLegBones =
	(
		legList.count = boneList.count
		
		if useLegBones.checked == true then
		(
			for i = 1 to boneList.count do
			(
				for l = 1 to legSelectionList.count do
				(
					if boneList[ i ].name == legSelectionList[ l ] then
					(
						legList[ i ] = 1
					)
				)
			)
		)
		else
		(
			for i = 1 to boneList.count do
			(
				legList[ i ] = 0
			)
		)
	)
	
	fn WriteSKL =
	(
		local filename = getSaveFileName caption:"Elite Force 2 SKL Export" types:"EF2 SKL File (*.skl)|*.skl|"
		if filename == undefined then
		(
			return false
		)
		
		local fp = createFile filename
		if fp == undefined then
		(
			format "Error creating file\n"
			return false
		)
		
		format "//////////////////////////////////////////////////////////////////////////\n" to:fp
		format "//\n" to:fp
		format "// Exported by TikiEngine Skeleton export, Version 1.00.\n" to:fp
		format "//\n" to:fp
		format "//////////////////////////////////////////////////////////////////////////\n\n" to:fp
		
		format "SKELETON\n" to:fp
		format "VERSION %\n\n" VERSION to:fp
		
		format "NUMMATERIALS %\n" model.nummaterials to:fp
		for m = 1 to model.nummaterials do
		(
			format "MATERIAL % %\n" model.materials[ m ] model.materialnames[ m ] to:fp
		)
		
		format "\n" to:fp
		format "NUMBONES %\n" boneList.count to:fp
		
		for f = 1 to boneList.count do
		(
			if legList[ f ] == 1 then
			(
				format "LEGBONE " to:fp
			)
			else
			(
				format "BONE " to:fp
			)
			format "% % %\n" (f-1) parentList[ f ] boneList[ f ].name to:fp
		)
		
		format "\n" to:fp
		
		format "NUMVERTS %\n" model.numverts to:fp
		for i = 1 to model.numverts do
		(
			format "VERT %\n" (i-1)	to:fp
			format "NORMAL " to:fp
				(WriteFloat6 model.verts[ i ].normal.x fp) 
				(WriteFloat6 model.verts[ i ].normal.y fp) 
				(WriteFloat6 model.verts[ i ].normal.z fp)
			format "\n" to:fp
			format "BONES %\n" model.verts[ i ].numbones to:fp
			
			for b = 1 to model.verts[ i ].numbones do
			(
				local blend = model.verts[ i ].blend[ b ]
				format "BONE % " blend.bone to:fp
					(WriteFloat6 blend.weight fp) 
					(WriteFloat6 blend.offset.x fp) 
					(WriteFloat6 blend.offset.y fp) 
					(WriteFloat6 blend.offset.z fp)
				format "\n" to:fp
			)
			format "\n" to:fp
		)
		
		format "NUMFACES %\n" model.numfaces to:fp
		for f = 1 to model.numfaces do
		(
			local tri = model.faces[ f ]
			format "TRI % % " (tri.id-1) ((tri.verts[ 1 ].vertindex-1) as integer) to:fp 
					(WriteFloat6 tri.verts[ 1 ].s fp)
					(WriteFloat6 tri.verts[ 1 ].t fp)
			format "% " ((tri.verts[ 2 ].vertindex-1) as integer) to:fp 
					(WriteFloat6 tri.verts[ 2 ].s fp)
					(WriteFloat6 tri.verts[ 2 ].t fp)
			format "% " ((tri.verts[ 3 ].vertindex-1) as integer) to:fp
					(WriteFloat6 tri.verts[ 3 ].s fp)
					(WriteFloat6 tri.verts[ 3 ].t fp)
			format "\n" to:fp
			
		)
		
		format "\n" to:fp
		
		format "FRAMERATE " to:fp
			(WriteFloat6 frameRate fp)
		format "\n" to:fp
				
		AddBoneFrames fp
		
		format "END\n" to:fp
		
		close fp
		
		ok	
	)
	
	---------------------------------------------------------
	--
	-- Morph Export functions
	--
	---------------------------------------------------------
	
	struct morphTarget
	(
		name,
		verts = #()
	)
	
	fn GetMorphData = 
	(
		max modify mode
		modPanel.SetCurrentObject morphSkinObj
		
		morphTargets = #()
		numMorphTargets = 0
		
		for i = 1 to 100 do
		(
			local curMorphTarget = morphTarget()
			if (WM3_MC_HasData morphModObj i) == true then
			(
				numMorphTargets += 1
				curMorphTarget.name = WM3_MC_GetName morphModObj i
				WM3_MC_SetValue morphModObj i 100.0
				local numVerts = morphBaseObj.numverts
				for v = 1 to numVerts do
				(
					local vert_pos = getVert morphBaseObj v
					local boneIndex = skinOps.GetVertexWeightBoneID morphSkinObj v 1
					local boneName = skinOps.GetBoneName morphSkinObj boneIndex 1
					local bone
					
					for b = 1 to boneList.count do
					(
						if boneList[ b ].name == boneName then
						(
							bone = boneList[ b ]
						)
					)
					
					local boneTM = bone.transform as matrix3
					local invBone = inverse boneTM
					
					local x = length boneTM.row1
					local y = length boneTM.row2
					local z = length boneTM.row3
					local s = [x, y, z] as point3
					
					local p = vert_pos * invBone
					
					x = p.x * s.x
					y = p.y * s.y
					z = p.z * s.z
					
					offset = [x, y, z]
					append curMorphTarget.verts offset
				)	
				WM3_MC_SetValue morphModObj i 0.0			
			)
			append morphTargets curMorphTarget
		)	
	)
	
	fn WriteMorphFile =
	(
		local filename = getSaveFileName filename:"morph.mph" types:"Skeleton MorphFile|*.mph|"
		if filename == undefined then
		(
			return false
		)
		
		local fp = createFile filename
		if fp == undefined then
		(
			return false
		)
		
		format "SKELETAL_MORPH_FILE_V1\n" to:fp
		format "targets: %\n" numMorphTargets to:fp
		
		for m = 1 to numMorphTargets do
		(
			local curMorphTarget = morphTargets[ m ]
			format "name: %\n" curMorphTarget.name to:fp
			format "verts: %\n" morphBaseObj.numverts to:fp
		
			for v = 1 to morphBaseObj.numverts do
			(
				format "v " to:fp
					(WriteFloat6 curMorphTarget.verts[ v ].x fp)
					(WriteFloat6 curMorphTarget.verts[ v ].y fp)
					(WriteFloat6 curMorphTarget.verts[ v ].z fp)
				format "\n" to:fp
			)
		)
		
		fclose fp	
	)

	on exportButton pressed do
	(
		InitStructures()
		CollectObjects()
		GetMaterials()
						
		GetLegBones()
		GetBoneParents()
		
		WriteSKL()
		
		if exportMorphTargets.checked == true then
		(
			GetMorphData()
			WriteMorphFile()
		)
	)
)